<?php

namespace App\Http\Controllers;

use App\Models\City;
use App\Models\Ride;
use App\Models\Route;
use App\Models\User;
use App\Models\Vehicle;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    /**
     * Display the landing page with available rides
     */
    public function index()
    {
        // Get popular routes
        $popularRoutes = Route::with(['fromCity', 'toCity'])
            ->where('is_popular', true)
            ->take(6)
            ->get();

        // Get upcoming rides with bookings and reviews
        $upcomingRides = Ride::with(['driver.reviewsReceived.reviewer', 'vehicle', 'route.fromCity', 'route.toCity', 'bookings.passenger'])
            ->upcoming()
            ->available()
            ->orderBy('departure_datetime', 'asc')
            ->take(6)
            ->get();

        // Get all cities for search dropdown
        $cities = City::where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('welcome', compact('popularRoutes', 'upcomingRides', 'cities'));
    }

    /**
     * Search for rides
     */
    public function searchRides(Request $request)
    {
        $query = Ride::with(['driver.reviewsReceived.reviewer', 'vehicle', 'route.fromCity', 'route.toCity', 'bookings.passenger'])
            ->upcoming()
            ->available();

        // Filter by route (support both 'from'/'to' and 'from_city'/'to_city')
        $fromCity = $request->from ?? $request->from_city;
        $toCity = $request->to ?? $request->to_city;
        
        if ($fromCity) {
            $query->whereHas('route', function ($q) use ($fromCity) {
                $q->where('from_city_id', $fromCity);
            });
        }
        
        if ($toCity) {
            $query->whereHas('route', function ($q) use ($toCity) {
                $q->where('to_city_id', $toCity);
            });
        }

        // Filter by date
        if ($request->filled('date')) {
            $query->whereDate('departure_datetime', $request->date);
        }

        // Filter by available seats
        if ($request->filled('seats')) {
            $query->where('available_seats', '>=', $request->seats);
        }

        $rides = $query->orderBy('departure_datetime', 'asc')->paginate(10);
        $cities = City::where('is_active', true)->orderBy('name')->get();

        return view('rides.search', compact('rides', 'cities'));
    }
    /**
     * Show the 'Offer a Ride' form
     */
    public function createRide()
    {
        $cities = City::where('is_active', true)->orderBy('name')->get();
        
        // Get user's vehicles (for demo, get first user's vehicles)
        $user = User::first();
        $vehicles = $user ? $user->vehicles : collect([]);
        
        return view('rides.create', compact('cities', 'vehicles'));
    }

    /**
     * Store a new ride
     */
    public function storeRide(Request $request)
    {
        $request->validate([
            'from_city' => 'required|exists:cities,id',
            'to_city' => 'required|exists:cities,id|different:from_city',
            'pickup_point' => 'required|string|max:255',
            'drop_point' => 'required|string|max:255',
            'date' => 'required|date|after_or_equal:today',
            'time' => 'required',
            'vehicle_id' => 'required|exists:vehicles,id',
            'seats' => 'required|integer|min:1|max:8',
            'price' => 'required|numeric|min:100',
            'amenities' => 'nullable|array',
            'preferences' => 'nullable|array',
            'notes' => 'nullable|string|max:500',
        ]);

        // Get or create route
        $route = Route::firstOrCreate(
            [
                'from_city_id' => $request->from_city,
                'to_city_id' => $request->to_city,
            ],
            [
                'distance_km' => 100, // Default - can be calculated
                'estimated_duration_minutes' => 120, // Default
                'base_price_per_seat' => $request->price / 280, // Convert from PKR
                'is_popular' => false,
            ]
        );

        // Get user (demo: first user)
        $user = User::first();
        $vehicle = Vehicle::find($request->vehicle_id);

        // Calculate arrival time (estimate 2 hours)
        $departureDateTime = \Carbon\Carbon::parse($request->date . ' ' . $request->time);
        $arrivalDateTime = $departureDateTime->copy()->addMinutes($route->estimated_duration_minutes ?? 120);

        // Create the ride
        $ride = Ride::create([
            'driver_id' => $user->id,
            'vehicle_id' => $request->vehicle_id,
            'route_id' => $route->id,
            'departure_datetime' => $departureDateTime,
            'arrival_datetime' => $arrivalDateTime,
            'available_seats' => $request->seats,
            'price_per_seat' => $request->price / 280, // Store in USD equivalent
            'full_car_price' => ($request->price / 280) * $request->seats * 0.9, // 10% discount for full car
            'ride_status' => 'scheduled',
            'pickup_point' => $request->pickup_point,
            'drop_point' => $request->drop_point,
            'amenities' => $request->amenities ?? [],
            'ride_preferences' => $request->preferences ?? [],
        ]);

        return redirect()->route('home')->with('success', 'Ride published successfully! Your ride from ' . $route->fromCity->name . ' to ' . $route->toCity->name . ' is now available for booking.');
    }

    /**
     * User Dashboard
     */
    public function dashboard()
    {
        // Check if user is logged in
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please login to view your dashboard.');
        }

        $user = auth()->user();
        $user->load(['bookingsAsPassenger.ride.route.fromCity', 'bookingsAsPassenger.ride.route.toCity', 'bookingsAsPassenger.ride.driver']);
        
        $upcomingBookings = collect([]);
        $pastBookings = collect([]);
        
        $upcomingBookings = $user->bookingsAsPassenger()
            ->whereHas('ride', function($q) {
                $q->where('departure_datetime', '>', now());
            })
            ->with(['ride.route.fromCity', 'ride.route.toCity', 'ride.driver'])
            ->take(3)
            ->get();
            
        $pastBookings = $user->bookingsAsPassenger()
            ->whereHas('ride', function($q) {
                $q->where('departure_datetime', '<', now());
            })
            ->with(['ride.route.fromCity', 'ride.route.toCity', 'ride.driver'])
            ->take(5)
            ->get();
        
        $stats = [
            'points' => 1250,
            'rides' => $user->bookingsAsPassenger->count(),
            'co2' => 48,
            'savings' => 12500,
        ];
        
        return view('dashboard', compact('user', 'upcomingBookings', 'pastBookings', 'stats'));
    }

    /**
     * Inbox / Messages
     */
    public function inbox()
    {
        return view('inbox');
    }

    /**
     * User Profile
     */
    public function profile()
    {
        // For demo, get first user or create default data
        $user = \App\Models\User::with(['vehicles', 'ridesAsDriver.route.fromCity', 'ridesAsDriver.route.toCity', 'bookingsAsPassenger.ride.route.fromCity', 'bookingsAsPassenger.ride.route.toCity', 'reviewsReceived.reviewer'])
            ->first();
        
        if (!$user) {
            // Demo user data
            $user = (object) [
                'id' => 1,
                'name' => 'Ahmed Khan',
                'email' => 'ahmed@example.com',
                'phone' => '+92 300 1234567',
                'avatar' => null,
                'user_type' => 'both',
                'is_verified' => true,
                'rating' => 4.8,
                'total_rides' => 24,
                'created_at' => now()->subMonths(6),
                'vehicles' => collect([]),
                'ridesAsDriver' => collect([]),
                'bookingsAsPassenger' => collect([]),
                'reviewsReceived' => collect([]),
            ];
        }

        $stats = [
            'total_rides' => $user->ridesAsDriver->count() + $user->bookingsAsPassenger->count(),
            'as_driver' => $user->ridesAsDriver->count(),
            'as_passenger' => $user->bookingsAsPassenger->count(),
            'rating' => $user->rating ?? 4.5,
        ];

        return view('profile', compact('user', 'stats'));
    }

    /**
     * Show the booking confirmation page
     */
    public function createBooking(Request $request)
    {
        $rideId = $request->query('ride');
        $ride = Ride::with(['driver', 'vehicle', 'route.fromCity', 'route.toCity'])->findOrFail($rideId);
        
        return view('bookings.create', compact('ride'));
    }

    /**
     * Store the booking
     */
    public function storeBooking(Request $request)
    {
        // Check if user is logged in
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please login to book a ride.');
        }

        $request->validate([
            'ride_id' => 'required|exists:rides,id',
            'seats' => 'required|integer|min:1|max:8',
            'phone' => 'required|string|max:20',
            'payment' => 'required|in:cash,jazzcash,easypaisa',
            'notes' => 'nullable|string|max:500',
        ]);

        $ride = Ride::findOrFail($request->ride_id);
        $user = auth()->user();
        
        // Check if seats available
        if ($ride->available_seats < $request->seats) {
            return back()->with('error', 'Not enough seats available');
        }

        // Calculate total price
        $serviceFee = 50;
        $totalPrice = ($ride->price_per_seat * 280 * $request->seats) + $serviceFee;

        // Create booking
        $booking = \App\Models\Booking::create([
            'ride_id' => $ride->id,
            'passenger_id' => $user->id,
            'booking_type' => 'seat',
            'number_of_seats' => $request->seats,
            'total_price' => $totalPrice / 280, // Store in USD equivalent
            'payment_status' => $request->payment == 'cash' ? 'pending' : 'paid',
            'booking_status' => 'confirmed',
            'pickup_location' => $ride->pickup_point,
            'drop_location' => $ride->drop_point,
            'passenger_notes' => $request->notes,
        ]);

        // Update available seats
        $ride->decrement('available_seats', $request->seats);

        return redirect()->route('dashboard')->with('success', 'Booking confirmed! Your ride from ' . $ride->route->fromCity->name . ' to ' . $ride->route->toCity->name . ' is booked for Rs ' . number_format($totalPrice) . '.');
    }

    /**
     * Driver Dashboard with earnings, bookings, etc.
     */
    public function driverDashboard()
    {
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please login to access driver dashboard.');
        }

        $user = auth()->user();
        
        // Check if user is driver
        if (!in_array($user->user_type, ['driver', 'both'])) {
            return redirect()->route('profile')->with('error', 'Switch to driver mode to access driver dashboard.');
        }

        // Get driver's rides
        $driverRides = Ride::where('driver_id', $user->id)->pluck('id');

        // Get bookings on driver's rides
        $recent_bookings = \App\Models\Booking::whereIn('ride_id', $driverRides)
            ->with(['ride.route.fromCity', 'ride.route.toCity', 'passenger'])
            ->latest()
            ->take(5)
            ->get();

        // Get vehicles
        $vehicles = Vehicle::where('driver_id', $user->id)->get();

        // Calculate stats
        $stats = [
            'total_rides' => Ride::where('driver_id', $user->id)->count(),
            'completed_rides' => Ride::where('driver_id', $user->id)->where('ride_status', 'completed')->count(),
            'pending_bookings' => \App\Models\Booking::whereIn('ride_id', $driverRides)->where('booking_status', 'pending')->count(),
        ];

        // Calculate earnings
        $totalEarnings = \App\Models\Booking::whereIn('ride_id', $driverRides)
            ->where('payment_status', 'paid')
            ->sum('total_price') * 280;

        $thisMonthEarnings = \App\Models\Booking::whereIn('ride_id', $driverRides)
            ->where('payment_status', 'paid')
            ->whereMonth('created_at', now()->month)
            ->sum('total_price') * 280;

        $lastMonthEarnings = \App\Models\Booking::whereIn('ride_id', $driverRides)
            ->where('payment_status', 'paid')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->sum('total_price') * 280;

        $pendingEarnings = \App\Models\Booking::whereIn('ride_id', $driverRides)
            ->where('payment_status', 'pending')
            ->sum('total_price') * 280;

        $earnings = [
            'total' => $totalEarnings,
            'this_month' => $thisMonthEarnings,
            'last_month' => $lastMonthEarnings,
            'pending' => $pendingEarnings,
        ];

        return view('driver.dashboard', compact('user', 'recent_bookings', 'vehicles', 'stats', 'earnings'));
    }
}
